<?php
require_once 'config/config.php';

if (!isLoggedIn()) {
    redirect('index.php');
}

$db = Database::getInstance()->getConnection();

// Get menu icons filtered by role
$role = getUserRole();
$allowed_menus = [];

if (isAdmin()) {
    $allowed_menus = ['dashboard', 'prediksi', 'peta', 'laporan', 'distribusi', 'optimasi', 'desa', 'users', 'depo', 'notifikasi', 'laporan_sistem', 'pengaturan', 'menu_icons', 'wa_broadcast', 'hak_akses'];
} elseif ($role === 'warga') {
    $allowed_menus = ['dashboard', 'prediksi', 'peta', 'laporan', 'distribusi', 'notifikasi'];
} elseif ($role === 'petugas_tangki') {
    $allowed_menus = ['dashboard', 'distribusi', 'notifikasi'];
} elseif ($role === 'pemdes') {
    $allowed_menus = ['dashboard', 'prediksi', 'laporan', 'distribusi', 'notifikasi'];
}

$allowed_keys = "'" . implode("','", $allowed_menus) . "'";
$stmt = $db->query("SELECT * FROM menu_icons WHERE is_active = 1 AND menu_key IN ($allowed_keys) ORDER BY urutan");
$menus = $stmt->fetchAll();

// Get all predictions (for warga) or recent (for admin)
if (getUserRole() === 'warga') {
    // Get all predictions for warga
    $stmt = $db->query("SELECT p.*, d.nama_dusun, d.latitude as dusun_lat, d.longitude as dusun_lng, des.nama_desa 
                        FROM prediksi_kekeringan p 
                        JOIN dusun d ON p.dusun_id = d.id 
                        JOIN desa des ON d.desa_id = des.id 
                        WHERE p.tanggal_prediksi >= CURDATE() 
                        ORDER BY p.skor_risiko DESC, p.created_at DESC");
    $all_predictions = $stmt->fetchAll();
    
    // Get user location
    $stmt = $db->prepare("SELECT latitude, longitude FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user_location = $stmt->fetch();
    
    // Get nearby predictions (within 10km radius)
    $nearby_predictions = [];
    if ($user_location && $user_location['latitude'] && $user_location['longitude']) {
        foreach ($all_predictions as $pred) {
            if ($pred['dusun_lat'] && $pred['dusun_lng']) {
                // Calculate distance
                $earth_radius = 6371; // km
                $lat_diff = deg2rad($pred['dusun_lat'] - $user_location['latitude']);
                $lng_diff = deg2rad($pred['dusun_lng'] - $user_location['longitude']);
                $a = sin($lat_diff/2) * sin($lat_diff/2) +
                     cos(deg2rad($user_location['latitude'])) * cos(deg2rad($pred['dusun_lat'])) *
                     sin($lng_diff/2) * sin($lng_diff/2);
                $c = 2 * atan2(sqrt($a), sqrt(1-$a));
                $distance = $earth_radius * $c;
                
                if ($distance <= 10) { // Within 10km
                    $pred['jarak_km'] = round($distance, 2);
                    $nearby_predictions[] = $pred;
                }
            }
        }
    }
    
    $recent_predictions = $all_predictions; // Show all for warga
    
    // Get statistics for warga
    $stats = [];
    $stats['total_prediksi'] = count($all_predictions);
    $stats['prediksi_kering'] = count(array_filter($all_predictions, fn($p) => $p['tingkat_risiko'] === 'kering'));
    $stats['prediksi_waspada'] = count(array_filter($all_predictions, fn($p) => $p['tingkat_risiko'] === 'waspada'));
    $stats['prediksi_aman'] = count(array_filter($all_predictions, fn($p) => $p['tingkat_risiko'] === 'aman'));
    $stmt = $db->prepare("SELECT COUNT(*) FROM laporan_warga WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $stats['laporan_saya'] = $stmt->fetchColumn();
} else {
    // Get recent predictions for admin/others
    $stmt = $db->query("SELECT p.*, d.nama_dusun, des.nama_desa 
                        FROM prediksi_kekeringan p 
                        JOIN dusun d ON p.dusun_id = d.id 
                        JOIN desa des ON d.desa_id = des.id 
                        WHERE p.tanggal_prediksi >= CURDATE() 
                        ORDER BY p.created_at DESC LIMIT 5");
    $recent_predictions = $stmt->fetchAll();
    $nearby_predictions = [];
    $user_location = null;
    
    // Get statistics for admin/others
    $stats = [];
    $stats['total_desa'] = $db->query("SELECT COUNT(*) FROM desa")->fetchColumn();
    $stats['total_dusun'] = $db->query("SELECT COUNT(*) FROM dusun")->fetchColumn();
    $stats['total_users'] = $db->query("SELECT COUNT(*) FROM users WHERE role = 'warga'")->fetchColumn();
    $stats['laporan_pending'] = $db->query("SELECT COUNT(*) FROM laporan_warga WHERE status = 'pending'")->fetchColumn();
    $stats['prediksi_kering'] = $db->query("SELECT COUNT(*) FROM prediksi_kekeringan WHERE tingkat_risiko = 'kering' AND tanggal_prediksi >= CURDATE()")->fetchColumn();
    $stats['tangki_terjadwal'] = $db->query("SELECT COUNT(*) FROM distribusi_tangki WHERE status = 'terjadwal'")->fetchColumn();
}

// Get notifications
$stmt = $db->prepare("SELECT * FROM notifikasi WHERE (user_id = ? OR user_id IS NULL) AND is_read = 0 ORDER BY created_at DESC LIMIT 5");
$stmt->execute([$_SESSION['user_id']]);
$notifications = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=5.0, user-scalable=yes">
    <title>Dashboard - <?php echo APP_NAME; ?></title>
    <?php if (!empty(APP_FAVICON)): ?>
    <link rel="icon" type="image/x-icon" href="<?php echo htmlspecialchars(APP_FAVICON); ?>">
    <?php endif; ?>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/responsive.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <?php if (getUserRole() === 'warga' && isset($user_location) && $user_location['latitude'] && $user_location['longitude']): ?>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <?php endif; ?>
</head>
<body>
    <!-- Mobile Menu Toggle -->
    <button class="menu-toggle" onclick="toggleSidebar()" id="menuToggle">
        <i class="fas fa-bars"></i>
    </button>
    
    <!-- Sidebar -->
    <?php include 'includes/sidebar.php'; ?>

    <!-- Main Content -->
    <main class="main-content">
        <!-- Header -->
        <header class="content-header">
            <div class="header-left">
                <h1>Dashboard</h1>
                <p>Selamat datang, <?php echo htmlspecialchars($_SESSION['nama_lengkap']); ?></p>
            </div>
            <div class="header-right">
                <div class="notifications-dropdown">
                    <button class="btn-icon" id="notifBtn">
                        <i class="fas fa-bell"></i>
                        <?php if (count($notifications) > 0): ?>
                        <span class="badge"><?php echo count($notifications); ?></span>
                        <?php endif; ?>
                    </button>
                    <div class="dropdown-menu" id="notifMenu">
                        <h3>Notifikasi</h3>
                        <?php if (empty($notifications)): ?>
                        <p class="empty">Tidak ada notifikasi</p>
                        <?php else: ?>
                        <?php foreach ($notifications as $notif): ?>
                        <div class="notification-item">
                            <div class="notif-icon <?php echo $notif['tipe']; ?>">
                                <i class="fas fa-<?php echo $notif['tipe'] === 'warning' ? 'exclamation-triangle' : ($notif['tipe'] === 'danger' ? 'times-circle' : 'info-circle'); ?>"></i>
                            </div>
                            <div class="notif-content">
                                <strong><?php echo htmlspecialchars($notif['judul']); ?></strong>
                                <p><?php echo htmlspecialchars($notif['pesan']); ?></p>
                                <small><?php echo formatDateTime($notif['created_at']); ?></small>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </header>

        <!-- Statistics -->
        <div class="stats-grid">
            <?php if (getUserRole() === 'warga'): ?>
            <div class="stat-card">
                <div class="stat-icon" style="background: #3498db;">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['total_prediksi']; ?></h3>
                    <p>Total Prediksi</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #e74c3c;">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['prediksi_kering']; ?></h3>
                    <p>Zona Kering</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #f39c12;">
                    <i class="fas fa-eye"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['prediksi_waspada']; ?></h3>
                    <p>Zona Waspada</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #2ecc71;">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['prediksi_aman']; ?></h3>
                    <p>Zona Aman</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #9b59b6;">
                    <i class="fas fa-file-alt"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['laporan_saya']; ?></h3>
                    <p>Laporan Saya</p>
                </div>
            </div>
            <?php else: ?>
            <div class="stat-card">
                <div class="stat-icon" style="background: #3498db;">
                    <i class="fas fa-building"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['total_desa']; ?></h3>
                    <p>Total Desa</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #2ecc71;">
                    <i class="fas fa-map-marker-alt"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['total_dusun']; ?></h3>
                    <p>Total Dusun</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #f39c12;">
                    <i class="fas fa-users"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['total_users']; ?></h3>
                    <p>Total Warga</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: #e74c3c;">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['prediksi_kering']; ?></h3>
                    <p>Zona Kering</p>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- Menu Grid -->
        <div class="menu-grid">
            <?php foreach ($menus as $menu): ?>
            <a href="<?php echo getMenuUrl($menu['menu_key']); ?>" class="menu-card">
                <div class="menu-icon" style="color: <?php echo $menu['icon_color']; ?>;">
                    <i class="<?php echo $menu['icon_class']; ?>"></i>
                </div>
                <h3><?php echo $menu['menu_label']; ?></h3>
            </a>
            <?php endforeach; ?>
        </div>

        <?php if (getUserRole() === 'warga'): ?>
        <!-- User Location Map & Condition -->
        <?php if ($user_location && $user_location['latitude'] && $user_location['longitude']): ?>
        <div class="content-section">
            <h2><i class="fas fa-map-marker-alt"></i> Lokasi & Kondisi Air Saya</h2>
            
            <!-- Find nearest dusun and its prediction -->
            <?php
            $nearest_dusun = null;
            $nearest_distance = null;
            $stmt = $db->query("SELECT d.*, des.nama_desa, p.tingkat_risiko, p.skor_risiko, p.rekomendasi
                               FROM dusun d
                               JOIN desa des ON d.desa_id = des.id
                               LEFT JOIN prediksi_kekeringan p ON d.id = p.dusun_id AND p.tanggal_prediksi >= CURDATE()
                               WHERE d.latitude IS NOT NULL AND d.longitude IS NOT NULL");
            $all_dusun = $stmt->fetchAll();
            
            foreach ($all_dusun as $dusun) {
                $earth_radius = 6371;
                $lat_diff = deg2rad($dusun['latitude'] - $user_location['latitude']);
                $lng_diff = deg2rad($dusun['longitude'] - $user_location['longitude']);
                $a = sin($lat_diff/2) * sin($lat_diff/2) +
                     cos(deg2rad($user_location['latitude'])) * cos(deg2rad($dusun['latitude'])) *
                     sin($lng_diff/2) * sin($lng_diff/2);
                $c = 2 * atan2(sqrt($a), sqrt(1-$a));
                $distance = $earth_radius * $c;
                
                if ($nearest_distance === null || $distance < $nearest_distance) {
                    $nearest_distance = $distance;
                    $nearest_dusun = $dusun;
                    $nearest_dusun['jarak_km'] = round($distance, 2);
                }
            }
            ?>
            
            <?php if ($nearest_dusun): ?>
            <div class="alert alert-<?php echo $nearest_dusun['tingkat_risiko'] === 'kering' ? 'danger' : ($nearest_dusun['tingkat_risiko'] === 'waspada' ? 'warning' : 'info'); ?>" style="margin-bottom: 20px;">
                <h3 style="margin-bottom: 10px;">
                    <i class="fas fa-<?php echo $nearest_dusun['tingkat_risiko'] === 'kering' ? 'exclamation-triangle' : ($nearest_dusun['tingkat_risiko'] === 'waspada' ? 'eye' : 'check-circle'); ?>"></i>
                    Kondisi Air di Daerah Terdekat
                </h3>
                <p><strong>Daerah:</strong> <?php echo htmlspecialchars($nearest_dusun['nama_desa']); ?> - <?php echo htmlspecialchars($nearest_dusun['nama_dusun']); ?></p>
                <p><strong>Jarak:</strong> <?php echo $nearest_dusun['jarak_km']; ?> km dari lokasi Anda</p>
                <?php if ($nearest_dusun['tingkat_risiko']): ?>
                <p><strong>Status:</strong> 
                    <span class="badge badge-<?php echo $nearest_dusun['tingkat_risiko'] === 'kering' ? 'danger' : ($nearest_dusun['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                        <?php echo strtoupper($nearest_dusun['tingkat_risiko']); ?>
                    </span>
                    (Skor: <?php echo number_format($nearest_dusun['skor_risiko'], 2); ?>)
                </p>
                <?php if ($nearest_dusun['rekomendasi']): ?>
                <p><strong>Rekomendasi:</strong> <?php echo htmlspecialchars($nearest_dusun['rekomendasi']); ?></p>
                <?php endif; ?>
                <?php else: ?>
                <p><strong>Status:</strong> Belum ada prediksi untuk daerah ini</p>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <div id="userMap" style="height: 400px; border-radius: 10px; margin-top: 15px;"></div>
            <div style="margin-top: 15px; display: flex; gap: 20px; flex-wrap: wrap;">
                <div>
                    <strong><i class="fas fa-map-marker-alt"></i> Koordinat Saya:</strong> 
                    <?php echo number_format($user_location['latitude'], 6); ?>, 
                    <?php echo number_format($user_location['longitude'], 6); ?>
                </div>
                <?php if (!empty($nearby_predictions)): ?>
                <div>
                    <strong><i class="fas fa-info-circle"></i> Daerah Terdekat:</strong> 
                    <?php echo count($nearby_predictions); ?> daerah dalam radius 10 km
                </div>
                <?php endif; ?>
                <?php if ($nearest_dusun): ?>
                <div>
                    <strong><i class="fas fa-location-arrow"></i> Daerah Terdekat:</strong> 
                    <?php echo htmlspecialchars($nearest_dusun['nama_dusun']); ?> 
                    (<?php echo $nearest_dusun['jarak_km']; ?> km)
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php else: ?>
        <div class="content-section">
            <div class="alert alert-warning">
                <i class="fas fa-exclamation-triangle"></i> 
                <strong>Lokasi GPS belum terdeteksi.</strong> 
                Koordinat GPS Anda belum tersimpan. Silakan update profil untuk menyimpan lokasi GPS Anda.
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Nearby Predictions -->
        <?php if (!empty($nearby_predictions)): ?>
        <div class="content-section">
            <h2><i class="fas fa-exclamation-triangle"></i> Prediksi Daerah Terdekat (Radius 10 km)</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Desa/Dusun</th>
                            <th>Jarak</th>
                            <th>Tingkat Risiko</th>
                            <th>Skor</th>
                            <th>Tanggal</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($nearby_predictions as $pred): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($pred['nama_desa']); ?> - <?php echo htmlspecialchars($pred['nama_dusun']); ?></td>
                            <td><?php echo number_format($pred['jarak_km'], 2); ?> km</td>
                            <td>
                                <span class="badge badge-<?php echo $pred['tingkat_risiko'] === 'kering' ? 'danger' : ($pred['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                                    <?php echo ucfirst($pred['tingkat_risiko']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($pred['skor_risiko'], 2); ?></td>
                            <td><?php echo formatDate($pred['tanggal_prediksi']); ?></td>
                            <td>
                                <a href="prediksi/detail.php?id=<?php echo $pred['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-eye"></i> Detail
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- All Predictions -->
        <div class="content-section">
            <h2><i class="fas fa-chart-line"></i> Semua Prediksi AI Kekeringan</h2>
            <p style="margin-bottom: 20px; color: #666;">
                <i class="fas fa-info-circle"></i> Informasi prediksi kondisi air untuk semua daerah
            </p>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Desa</th>
                            <th>Dusun</th>
                            <th>Tanggal Prediksi</th>
                            <th>Tingkat Risiko</th>
                            <th>Skor Risiko</th>
                            <th>Rekomendasi</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($recent_predictions)): ?>
                        <tr>
                            <td colspan="7" class="text-center">Belum ada prediksi. Admin akan membuat prediksi secara berkala.</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($recent_predictions as $pred): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($pred['nama_desa']); ?></td>
                            <td><?php echo htmlspecialchars($pred['nama_dusun']); ?></td>
                            <td><?php echo formatDate($pred['tanggal_prediksi']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $pred['tingkat_risiko'] === 'kering' ? 'danger' : ($pred['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                                    <?php echo ucfirst($pred['tingkat_risiko']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($pred['skor_risiko'], 2); ?></td>
                            <td>
                                <?php if ($pred['tingkat_risiko'] === 'kering'): ?>
                                <span style="color: #e74c3c;"><i class="fas fa-exclamation-triangle"></i> Segera hemat air</span>
                                <?php elseif ($pred['tingkat_risiko'] === 'waspada'): ?>
                                <span style="color: #f39c12;"><i class="fas fa-eye"></i> Mulai waspada</span>
                                <?php else: ?>
                                <span style="color: #2ecc71;"><i class="fas fa-check-circle"></i> Kondisi normal</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="prediksi/detail.php?id=<?php echo $pred['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-eye"></i> Detail
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php else: ?>
        <!-- Recent Predictions (Admin/Others) -->
        <div class="content-section">
            <h2>Prediksi Terbaru</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Dusun</th>
                            <th>Tanggal Prediksi</th>
                            <th>Tingkat Risiko</th>
                            <th>Skor</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($recent_predictions)): ?>
                        <tr>
                            <td colspan="5" class="text-center">Belum ada prediksi</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($recent_predictions as $pred): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($pred['nama_dusun']); ?></td>
                            <td><?php echo formatDate($pred['tanggal_prediksi']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $pred['tingkat_risiko'] === 'kering' ? 'danger' : ($pred['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                                    <?php echo ucfirst($pred['tingkat_risiko']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($pred['skor_risiko'], 2); ?></td>
                            <td>
                                <a href="prediksi/detail.php?id=<?php echo $pred['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-eye"></i> Detail
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
    </main>

    <script src="assets/js/main.js"></script>
    <script>
    function toggleSidebar() {
        document.querySelector('.sidebar').classList.toggle('active');
    }
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
        const sidebar = document.querySelector('.sidebar');
        const toggle = document.querySelector('.menu-toggle');
        
        if (window.innerWidth <= 768) {
            if (!sidebar.contains(event.target) && !toggle.contains(event.target) && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
            }
        }
    });
    
    <?php if (getUserRole() === 'warga' && isset($user_location) && $user_location['latitude'] && $user_location['longitude']): ?>
    // Initialize user location map
    const userMap = L.map('userMap').setView([<?php echo $user_location['latitude']; ?>, <?php echo $user_location['longitude']; ?>], 13);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(userMap);
    
    // Add user marker
    const userMarker = L.marker([<?php echo $user_location['latitude']; ?>, <?php echo $user_location['longitude']; ?>], {
        icon: L.divIcon({
            className: 'user-marker',
            html: '<div style="background: #3498db; width: 30px; height: 30px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 10px rgba(0,0,0,0.3);"></div>',
            iconSize: [30, 30],
            iconAnchor: [15, 15]
        })
    }).addTo(userMap);
    userMarker.bindPopup('<strong>Lokasi Saya</strong><br><?php echo htmlspecialchars($_SESSION['nama_lengkap']); ?>').openPopup();
    
    // Add nearby predictions markers
    <?php if (!empty($nearby_predictions)): ?>
    <?php foreach ($nearby_predictions as $pred): ?>
    <?php if ($pred['dusun_lat'] && $pred['dusun_lng']): ?>
    const color<?php echo $pred['id']; ?> = '<?php echo $pred['tingkat_risiko'] === 'kering' ? '#e74c3c' : ($pred['tingkat_risiko'] === 'waspada' ? '#f39c12' : '#2ecc71'); ?>';
    L.circleMarker([<?php echo $pred['dusun_lat']; ?>, <?php echo $pred['dusun_lng']; ?>], {
        radius: 8,
        fillColor: color<?php echo $pred['id']; ?>,
        color: '#fff',
        weight: 2,
        opacity: 1,
        fillOpacity: 0.8
    }).addTo(userMap).bindPopup(
        '<strong><?php echo htmlspecialchars($pred['nama_dusun']); ?></strong><br>' +
        'Status: <?php echo ucfirst($pred['tingkat_risiko']); ?><br>' +
        'Jarak: <?php echo number_format($pred['jarak_km'], 2); ?> km<br>' +
        'Skor: <?php echo number_format($pred['skor_risiko'], 2); ?>'
    );
    <?php endif; ?>
    <?php endforeach; ?>
    <?php endif; ?>
    
    // Add nearest dusun marker
    <?php if (isset($nearest_dusun) && $nearest_dusun && $nearest_dusun['latitude'] && $nearest_dusun['longitude']): ?>
    L.marker([<?php echo $nearest_dusun['latitude']; ?>, <?php echo $nearest_dusun['longitude']; ?>], {
        icon: L.divIcon({
            className: 'nearest-marker',
            html: '<div style="background: #9b59b6; width: 25px; height: 25px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 10px rgba(0,0,0,0.3);"></div>',
            iconSize: [25, 25],
            iconAnchor: [12, 12]
        })
    }).addTo(userMap).bindPopup(
        '<strong>Daerah Terdekat</strong><br>' +
        '<?php echo htmlspecialchars($nearest_dusun['nama_dusun']); ?><br>' +
        'Jarak: <?php echo $nearest_dusun['jarak_km']; ?> km'
    );
    <?php endif; ?>
    <?php endif; ?>
    </script>
</body>
</html>

<?php
function getMenuUrl($key) {
    $urls = [
        'dashboard' => 'dashboard.php',
        'prediksi' => 'prediksi/index.php',
        'peta' => 'peta/index.php',
        'laporan' => 'laporan/index.php',
        'distribusi' => (isAdmin() || getUserRole() === 'pemdes') ? 'admin/distribusi.php' : (getUserRole() === 'petugas_tangki' ? 'petugas/index.php' : 'distribusi/warga.php'),
        'optimasi' => 'admin/optimasi.php',
        'desa' => 'admin/desa.php',
        'users' => 'admin/users.php',
        'depo' => 'admin/depo.php',
        'notifikasi' => 'notifikasi/index.php',
        'laporan_sistem' => 'admin/laporan.php',
        'pengaturan' => 'admin/pengaturan.php',
        'menu_icons' => 'admin/menu_icons.php',
        'wa_broadcast' => 'admin/wa_broadcast.php',
        'hak_akses' => 'admin/hak_akses.php'
    ];
    return $urls[$key] ?? 'dashboard.php';
}
?>

