<?php
/**
 * Error Check & Synchronization Verification
 * Run this file to check for errors and synchronization issues
 */

require_once '../config/config.php';

if (!isAdmin()) {
    die('Access denied');
}

$db = Database::getInstance()->getConnection();
$errors = [];
$warnings = [];
$success = [];

// Check 1: Database tables
$required_tables = [
    'users', 'desa', 'dusun', 'data_curah_hujan', 'laporan_warga', 
    'prediksi_kekeringan', 'notifikasi', 'menu_icons', 'distribusi_tangki',
    'depo_pengiriman', 'wa_config', 'wa_broadcast', 'wa_broadcast_log',
    'app_config', 'role_permissions'
];

foreach ($required_tables as $table) {
    try {
        $stmt = $db->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() === 0) {
            $errors[] = "Table '$table' tidak ditemukan";
        } else {
            $success[] = "Table '$table' OK";
        }
    } catch (Exception $e) {
        $errors[] = "Error checking table '$table': " . $e->getMessage();
    }
}

// Check 2: Required columns in users table
$required_columns = [
    'id', 'username', 'password', 'nama_lengkap', 'email', 'no_hp',
    'no_ktp', 'provinsi', 'kabupaten', 'kelurahan', 'dusun_id', 'rt', 'rw',
    'latitude', 'longitude', 'role', 'status'
];

try {
    $stmt = $db->query("SHOW COLUMNS FROM users");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    foreach ($required_columns as $col) {
        if (!in_array($col, $columns)) {
            $errors[] = "Column 'users.$col' tidak ditemukan";
        }
    }
    $success[] = "Users table columns OK";
} catch (Exception $e) {
    $errors[] = "Error checking users columns: " . $e->getMessage();
}

// Check 3: App config
try {
    $stmt = $db->query("SELECT config_key FROM app_config WHERE config_key IN ('app_name', 'app_desc', 'app_logo', 'app_favicon')");
    $configs = $stmt->fetchAll(PDO::FETCH_COLUMN);
    $required_configs = ['app_name', 'app_desc', 'app_logo', 'app_favicon'];
    foreach ($required_configs as $key) {
        if (!in_array($key, $configs)) {
            $warnings[] = "Config '$key' belum diatur (akan menggunakan default)";
        }
    }
    if (count($configs) >= 2) {
        $success[] = "App config OK";
    }
} catch (Exception $e) {
    $warnings[] = "App config table mungkin belum dibuat: " . $e->getMessage();
}

// Check 4: File permissions
$upload_dirs = [
    'uploads/',
    'uploads/laporan/',
    'uploads/broadcast/',
    'uploads/logo/'
];

foreach ($upload_dirs as $dir) {
    $path = BASE_PATH . $dir;
    if (!is_dir($path)) {
        $warnings[] = "Directory '$dir' tidak ada (akan dibuat otomatis saat upload)";
    } elseif (!is_writable($path)) {
        $errors[] = "Directory '$dir' tidak dapat ditulis (perlu chmod 755 atau 777)";
    } else {
        $success[] = "Directory '$dir' OK";
    }
}

// Check 5: Required files
$required_files = [
    'config/config.php',
    'config/database.php',
    'index.php',
    'dashboard.php',
    'auth/login.php',
    'auth/register.php',
    'auth/logout.php',
    'includes/sidebar.php',
    'admin/users.php',
    'admin/pengaturan.php',
    'admin/hak_akses.php'
];

foreach ($required_files as $file) {
    $path = BASE_PATH . $file;
    if (!file_exists($path)) {
        $errors[] = "File '$file' tidak ditemukan";
    } else {
        $success[] = "File '$file' OK";
    }
}

// Check 6: Constants
$required_constants = ['APP_NAME', 'APP_DESC', 'BASE_PATH', 'APP_URL'];
foreach ($required_constants as $const) {
    if (!defined($const)) {
        $errors[] = "Constant '$const' tidak terdefinisi";
    } else {
        $success[] = "Constant '$const' OK";
    }
}

// Check 7: Foreign keys
try {
    $stmt = $db->query("SELECT COUNT(*) FROM users WHERE dusun_id IS NOT NULL AND dusun_id NOT IN (SELECT id FROM dusun)");
    $orphan_users = $stmt->fetchColumn();
    if ($orphan_users > 0) {
        $warnings[] = "Ditemukan $orphan_users user dengan dusun_id yang tidak valid";
    }
    
    $stmt = $db->query("SELECT COUNT(*) FROM distribusi_tangki WHERE depo_id IS NOT NULL AND depo_id NOT IN (SELECT id FROM depo_pengiriman)");
    $orphan_distribusi = $stmt->fetchColumn();
    if ($orphan_distribusi > 0) {
        $warnings[] = "Ditemukan $orphan_distribusi distribusi dengan depo_id yang tidak valid";
    }
    
    $success[] = "Foreign key integrity OK";
} catch (Exception $e) {
    $warnings[] = "Tidak dapat memeriksa foreign keys: " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Error Check - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fas fa-bug"></i> Error Check & Synchronization</h1>
                <p>Pemeriksaan error dan sinkronisasi sistem</p>
            </div>
            <a href="pengaturan.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Kembali
            </a>
        </header>

        <div class="content-section">
            <h2>Hasil Pemeriksaan</h2>
            
            <?php if (!empty($errors)): ?>
            <div class="alert alert-danger" style="margin-bottom: 20px;">
                <h3><i class="fas fa-exclamation-triangle"></i> Errors (<?php echo count($errors); ?>)</h3>
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($warnings)): ?>
            <div class="alert alert-warning" style="margin-bottom: 20px;">
                <h3><i class="fas fa-exclamation-circle"></i> Warnings (<?php echo count($warnings); ?>)</h3>
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <?php foreach ($warnings as $warning): ?>
                    <li><?php echo htmlspecialchars($warning); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($success)): ?>
            <div class="alert alert-success">
                <h3><i class="fas fa-check-circle"></i> Success (<?php echo count($success); ?>)</h3>
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <?php foreach ($success as $msg): ?>
                    <li><?php echo htmlspecialchars($msg); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 30px; padding: 20px; background: #f8f9fa; border-radius: 8px;">
                <h3>Summary</h3>
                <p><strong>Total Errors:</strong> <?php echo count($errors); ?></p>
                <p><strong>Total Warnings:</strong> <?php echo count($warnings); ?></p>
                <p><strong>Total Success:</strong> <?php echo count($success); ?></p>
                
                <?php if (empty($errors)): ?>
                <div class="alert alert-success" style="margin-top: 20px;">
                    <i class="fas fa-check-circle"></i> <strong>Sistem tidak memiliki error!</strong>
                </div>
                <?php else: ?>
                <div class="alert alert-danger" style="margin-top: 20px;">
                    <i class="fas fa-exclamation-triangle"></i> <strong>Ditemukan <?php echo count($errors); ?> error yang perlu diperbaiki!</strong>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <script src="../assets/js/main.js"></script>
</body>
</html>

