<?php
require_once '../config/config.php';

if (getUserRole() !== 'petugas_tangki') {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle upload bukti
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_bukti') {
    $id = intval($_POST['id'] ?? 0);
    $foto = sanitize($_POST['foto'] ?? '');
    $latitude = floatval($_POST['latitude'] ?? 0);
    $longitude = floatval($_POST['longitude'] ?? 0);
    
    if ($id > 0 && !empty($foto)) {
        $stmt = $db->prepare("UPDATE distribusi_tangki SET foto_bukti = ?, latitude_bukti = ?, longitude_bukti = ? WHERE id = ? AND petugas_id = ?");
        if ($stmt->execute([$foto, $latitude, $longitude, $id, $_SESSION['user_id']])) {
            $_SESSION['success'] = 'Bukti pengiriman berhasil diupload';
        } else {
            $_SESSION['error'] = 'Gagal upload bukti';
        }
    } else {
        $_SESSION['error'] = 'Data tidak lengkap';
    }
    
    redirect('index.php');
}

// Get my distributions
$stmt = $db->prepare("SELECT dt.*, dus.nama_dusun, des.nama_desa 
                    FROM distribusi_tangki dt
                    JOIN dusun dus ON dt.dusun_id = dus.id
                    JOIN desa des ON dus.desa_id = des.id
                    WHERE dt.petugas_id = ? AND dt.status IN ('dalam_perjalanan', 'selesai')
                    ORDER BY dt.tanggal_jadwal DESC");
$stmt->execute([$_SESSION['user_id']]);
$my_distributions = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laporan Petugas - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <aside class="sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-tint"></i> <?php echo APP_NAME; ?></h2>
        </div>
        <nav class="sidebar-nav">
            <a href="../dashboard.php" class="nav-item">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>
            <a href="index.php" class="nav-item active">
                <i class="fas fa-truck"></i>
                <span>Laporan Pengiriman</span>
            </a>
            <a href="../auth/logout.php" class="nav-item">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </nav>
    </aside>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fas fa-truck"></i> Laporan Pengiriman Air</h1>
                <p>Upload bukti pengiriman dengan foto dan koordinat GPS</p>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <div class="content-section">
            <h2>Distribusi Saya</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Tanggal</th>
                            <th>Desa/Dusun</th>
                            <th>Volume</th>
                            <th>Status</th>
                            <th>Bukti</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($my_distributions)): ?>
                        <tr>
                            <td colspan="6" class="text-center">Belum ada distribusi yang ditugaskan</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($my_distributions as $dist): ?>
                        <tr>
                            <td><?php echo formatDate($dist['tanggal_jadwal']); ?></td>
                            <td><?php echo htmlspecialchars($dist['nama_desa']); ?> - <?php echo htmlspecialchars($dist['nama_dusun']); ?></td>
                            <td><?php echo number_format($dist['volume_liter'] ?? 0); ?> L</td>
                            <td>
                                <span class="badge badge-<?php echo $dist['status'] === 'selesai' ? 'success' : 'warning'; ?>">
                                    <?php echo ucfirst(str_replace('_', ' ', $dist['status'])); ?>
                                </span>
                            </td>
                            <td>
                                <?php if (!empty($dist['foto_bukti'])): ?>
                                <span class="badge badge-success">
                                    <i class="fas fa-check"></i> Sudah Upload
                                </span>
                                <?php else: ?>
                                <span class="badge badge-warning">
                                    <i class="fas fa-times"></i> Belum Upload
                                </span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (empty($dist['foto_bukti']) && $dist['status'] === 'dalam_perjalanan'): ?>
                                <button class="btn btn-sm btn-primary" onclick="uploadBukti(<?php echo $dist['id']; ?>, '<?php echo htmlspecialchars($dist['nama_dusun']); ?>')">
                                    <i class="fas fa-camera"></i> Upload Bukti
                                </button>
                                <?php elseif (!empty($dist['foto_bukti'])): ?>
                                <a href="detail.php?id=<?php echo $dist['id']; ?>" class="btn btn-sm btn-success">
                                    <i class="fas fa-eye"></i> Lihat
                                </a>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Upload Bukti Modal -->
    <div id="uploadModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('uploadModal').style.display='none'">&times;</span>
            <h2>Upload Bukti Pengiriman</h2>
            <form method="POST" class="form-container" id="buktiForm">
                <input type="hidden" name="action" value="upload_bukti">
                <input type="hidden" name="id" id="bukti_id">
                <input type="hidden" name="foto" id="bukti_foto">
                <input type="hidden" name="latitude" id="bukti_latitude">
                <input type="hidden" name="longitude" id="bukti_longitude">
                
                <div class="form-group">
                    <label>Upload Foto Bukti</label>
                    <div style="display: flex; gap: 10px; margin-bottom: 10px;">
                        <input type="file" id="bukti_foto_input" accept="image/*" capture="environment" style="flex: 1;">
                        <button type="button" class="btn btn-secondary" onclick="document.getElementById('bukti_foto_input').click()">
                            <i class="fas fa-camera"></i> Ambil Foto
                        </button>
                    </div>
                    <small style="color: #666;">Ambil foto langsung dari kamera atau pilih dari galeri</small>
                    <div id="bukti_preview" style="margin-top: 10px; display: none;">
                        <img id="bukti_preview_img" src="" alt="Preview" style="max-width: 100%; max-height: 200px; border-radius: 8px;">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeBuktiFoto()" style="margin-top: 10px;">
                            <i class="fas fa-times"></i> Hapus Foto
                        </button>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Koordinat GPS</label>
                    <div class="form-row">
                        <div class="form-group">
                            <label>Latitude</label>
                            <input type="text" id="bukti_lat_display" readonly style="background: #f5f5f5;">
                        </div>
                        <div class="form-group">
                            <label>Longitude</label>
                            <input type="text" id="bukti_lng_display" readonly style="background: #f5f5f5;">
                        </div>
                    </div>
                    <button type="button" class="btn btn-sm btn-primary" onclick="getBuktiLocation()">
                        <i class="fas fa-map-marker-alt"></i> Ambil Koordinat
                    </button>
                    <small style="color: #666; display: block; margin-top: 5px;">Koordinat akan diambil otomatis dari GPS perangkat Anda</small>
                </div>
                
                <button type="submit" class="btn btn-primary btn-block">
                    <i class="fas fa-upload"></i> Upload Bukti
                </button>
            </form>
        </div>
    </div>

    <script src="../assets/js/main.js"></script>
    <script src="../laporan/upload.php"></script>
    <script>
    let currentBuktiFoto = null;
    
    // Get GPS Location
    function getBuktiLocation() {
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const lat = position.coords.latitude;
                    const lng = position.coords.longitude;
                    
                    document.getElementById('bukti_latitude').value = lat;
                    document.getElementById('bukti_longitude').value = lng;
                    document.getElementById('bukti_lat_display').value = lat.toFixed(6);
                    document.getElementById('bukti_lng_display').value = lng.toFixed(6);
                    
                    showToast('Koordinat berhasil diambil', 'success');
                },
                function(error) {
                    showToast('Gagal mengambil koordinat: ' + error.message, 'danger');
                }
            );
        } else {
            showToast('GPS tidak didukung di perangkat ini', 'warning');
        }
    }
    
    function uploadBukti(id, dusunName) {
        document.getElementById('uploadModal').style.display = 'block';
        document.getElementById('bukti_id').value = id;
        document.getElementById('buktiForm').reset();
        document.getElementById('bukti_preview').style.display = 'none';
        currentBuktiFoto = null;
        
        // Auto get location
        getBuktiLocation();
    }
    
    // Handle foto upload
    document.getElementById('bukti_foto_input').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const formData = new FormData();
            formData.append('foto', file);
            
            const preview = document.getElementById('bukti_preview');
            preview.innerHTML = '<div class="loading"></div> Uploading...';
            preview.style.display = 'block';
            
            fetch('../laporan/upload.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    currentBuktiFoto = data.path;
                    document.getElementById('bukti_foto').value = data.path;
                    document.getElementById('bukti_preview_img').src = '../' + data.path;
                    preview.innerHTML = `
                        <img id="bukti_preview_img" src="../${data.path}" alt="Preview" style="max-width: 100%; max-height: 200px; border-radius: 8px;">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeBuktiFoto()" style="margin-top: 10px;">
                            <i class="fas fa-times"></i> Hapus Foto
                        </button>
                    `;
                    showToast('Foto berhasil diupload', 'success');
                } else {
                    preview.style.display = 'none';
                    showToast(data.message || 'Gagal upload foto', 'danger');
                }
            })
            .catch(error => {
                preview.style.display = 'none';
                showToast('Error: ' + error.message, 'danger');
            });
        }
    });
    
    function removeBuktiFoto() {
        currentBuktiFoto = null;
        document.getElementById('bukti_foto').value = '';
        document.getElementById('bukti_foto_input').value = '';
        document.getElementById('bukti_preview').style.display = 'none';
    }
    
    // Form submit
    document.getElementById('buktiForm').addEventListener('submit', function(e) {
        if (!document.getElementById('bukti_foto').value) {
            e.preventDefault();
            showToast('Foto harus diupload', 'danger');
            return;
        }
        
        if (!document.getElementById('bukti_latitude').value || !document.getElementById('bukti_longitude').value) {
            e.preventDefault();
            if (confirm('Koordinat GPS belum diambil. Lanjutkan tanpa koordinat?')) {
                document.getElementById('bukti_latitude').value = 0;
                document.getElementById('bukti_longitude').value = 0;
                this.submit();
            }
        }
    });
    
    window.onclick = function(event) {
        if (event.target == document.getElementById('uploadModal')) {
            document.getElementById('uploadModal').style.display = 'none';
        }
    }
    </script>
    <style>
    .modal {
        display: none;
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        overflow-y: auto;
    }
    
    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 15px;
        width: 90%;
        max-width: 600px;
    }
    
    .close {
        float: right;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
        color: #999;
    }
    
    #bukti_preview img {
        border: 2px solid #ddd;
        padding: 5px;
    }
    </style>
</body>
</html>

