<?php
require_once '../config/config.php';

if (!isAdmin() && getUserRole() !== 'pemdes') {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle optimize
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'optimize') {
    $tanggal = sanitize($_POST['tanggal'] ?? date('Y-m-d'));
    $depo_id = intval($_POST['depo_id'] ?? 0);
    
    // Get pending distributions with prediction data
    $stmt = $db->prepare("SELECT dt.*, dus.latitude as dusun_lat, dus.longitude as dusun_lng, 
                         dus.nama_dusun, des.nama_desa,
                         p.tingkat_risiko, p.skor_risiko,
                         (SELECT COUNT(*) FROM laporan_warga lw WHERE lw.dusun_id = dus.id AND lw.status = 'verified' AND lw.tanggal >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)) as jumlah_laporan
                         FROM distribusi_tangki dt
                         JOIN dusun dus ON dt.dusun_id = dus.id
                         JOIN desa des ON dus.desa_id = des.id
                         LEFT JOIN prediksi_kekeringan p ON dus.id = p.dusun_id AND p.tanggal_prediksi >= CURDATE()
                         WHERE dt.status = 'disetujui' AND dt.tanggal_jadwal = ? AND dus.latitude IS NOT NULL AND dus.longitude IS NOT NULL");
    $stmt->execute([$tanggal]);
    $distributions = $stmt->fetchAll();
    
    if (empty($distributions)) {
        $_SESSION['error'] = 'Tidak ada distribusi yang perlu dioptimasi';
        redirect('optimasi.php');
    }
    
    // Get depo
    $stmt = $db->prepare("SELECT * FROM depo_pengiriman WHERE id = ? AND is_active = 1");
    $stmt->execute([$depo_id]);
    $depo = $stmt->fetch();
    
    if (!$depo) {
        $_SESSION['error'] = 'Depo tidak ditemukan';
        redirect('optimasi.php');
    }
    
    // Calculate distance and optimize route
    $depo_lat = floatval($depo['latitude']);
    $depo_lng = floatval($depo['longitude']);
    
    // Calculate priority score for each distribution
    // Priority = (Condition Score * 0.5) + (Distance Score * 0.3) + (Report Score * 0.2)
    foreach ($distributions as &$dist) {
        $dusun_lat = floatval($dist['dusun_lat']);
        $dusun_lng = floatval($dist['dusun_lng']);
        
        // Calculate distance from depo
        $earth_radius = 6371; // km
        $lat_diff = deg2rad($dusun_lat - $depo_lat);
        $lng_diff = deg2rad($dusun_lng - $depo_lng);
        $a = sin($lat_diff/2) * sin($lat_diff/2) +
             cos(deg2rad($depo_lat)) * cos(deg2rad($dusun_lat)) *
             sin($lng_diff/2) * sin($lng_diff/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $distance = $earth_radius * $c;
        
        $dist['jarak_km'] = round($distance, 2);
        
        // Condition Score (based on prediction)
        $condition_score = 0;
        if ($dist['tingkat_risiko'] === 'kering') {
            $condition_score = 100; // Highest priority
        } elseif ($dist['tingkat_risiko'] === 'waspada') {
            $condition_score = 60;
        } elseif ($dist['tingkat_risiko'] === 'aman') {
            $condition_score = 20;
        } else {
            $condition_score = 30; // No prediction, medium priority
        }
        
        // Distance Score (closer = higher score, max 50km = 0 score)
        $max_distance = 50; // km
        $distance_score = max(0, 100 - ($distance / $max_distance * 100));
        
        // Report Score (more reports = higher priority)
        $report_count = intval($dist['jumlah_laporan'] ?? 0);
        $report_score = min(100, $report_count * 20); // Max 5 reports = 100
        
        // Calculate priority score (higher = more urgent)
        $priority_score = ($condition_score * 0.5) + ($distance_score * 0.3) + ($report_score * 0.2);
        
        $dist['condition_score'] = $condition_score;
        $dist['distance_score'] = $distance_score;
        $dist['report_score'] = $report_score;
        $dist['priority_score'] = round($priority_score, 2);
    }
    
    // Sort by priority score (highest first) - prioritize urgent conditions and nearby locations
    usort($distributions, function($a, $b) {
        // First sort by priority score (descending)
        if ($a['priority_score'] != $b['priority_score']) {
            return $b['priority_score'] <=> $a['priority_score'];
        }
        // If same priority, sort by distance (nearest first)
        return $a['jarak_km'] <=> $b['jarak_km'];
    });
    
    // Optimize route using Nearest Neighbor Algorithm after priority sorting
    // This ensures we visit locations in priority order but minimize travel distance
    $optimized_route = [];
    $visited = [];
    $current_lat = $depo_lat;
    $current_lng = $depo_lng;
    $urutan = 1;
    
    // Start with highest priority items
    while (count($optimized_route) < count($distributions)) {
        $nearest = null;
        $nearest_distance = null;
        $nearest_index = null;
        
        // Find nearest unvisited location from current position
        foreach ($distributions as $index => $dist) {
            if (isset($visited[$index])) continue;
            
            $dusun_lat = floatval($dist['dusun_lat']);
            $dusun_lng = floatval($dist['dusun_lng']);
            
            $lat_diff = deg2rad($dusun_lat - $current_lat);
            $lng_diff = deg2rad($dusun_lng - $current_lng);
            $a = sin($lat_diff/2) * sin($lat_diff/2) +
                 cos(deg2rad($current_lat)) * cos(deg2rad($dusun_lat)) *
                 sin($lng_diff/2) * sin($lng_diff/2);
            $c = 2 * atan2(sqrt($a), sqrt(1-$a));
            $distance = $earth_radius * $c;
            
            // For first item, prefer highest priority
            // For subsequent items, prefer nearest but still consider priority
            if ($nearest === null || 
                (count($optimized_route) === 0 && $dist['priority_score'] > ($nearest['priority_score'] ?? 0)) ||
                (count($optimized_route) > 0 && $distance < $nearest_distance)) {
                $nearest = $dist;
                $nearest_distance = $distance;
                $nearest_index = $index;
            }
        }
        
        if ($nearest === null) break;
        
        $visited[$nearest_index] = true;
        $optimized_route[] = $nearest;
        
        // Calculate segment distance and BBM
        $segment_distance = $nearest_distance;
        $bbm_per_km = 0.2; // liter per km
        $estimasi_bbm = $segment_distance * $bbm_per_km;
        $biaya_bbm = $estimasi_bbm * 10000; // Assuming 10,000 per liter
        
        // Update distribution with optimized route order
        $stmt = $db->prepare("UPDATE distribusi_tangki SET 
                             depo_id = ?, jarak_km = ?, estimasi_bbm_liter = ?, 
                             biaya_bbm = ?, urutan_rute = ? 
                             WHERE id = ?");
        $stmt->execute([
            $depo_id,
            round($segment_distance, 2),
            round($estimasi_bbm, 2),
            round($biaya_bbm, 2),
            $urutan,
            $nearest['id']
        ]);
        
        $current_lat = floatval($nearest['dusun_lat']);
        $current_lng = floatval($nearest['dusun_lng']);
        $urutan++;
    }
    
    // Calculate total distance
    $total_distance = 0;
    $previous_lat = $depo_lat;
    $previous_lng = $depo_lng;
    
    foreach ($optimized_route as $dist) {
        $lat_diff = deg2rad(floatval($dist['dusun_lat']) - $previous_lat);
        $lng_diff = deg2rad(floatval($dist['dusun_lng']) - $previous_lng);
        $a = sin($lat_diff/2) * sin($lat_diff/2) +
             cos(deg2rad($previous_lat)) * cos(deg2rad(floatval($dist['dusun_lat']))) *
             sin($lng_diff/2) * sin($lng_diff/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $segment_distance = $earth_radius * $c;
        $total_distance += $segment_distance;
        
        $previous_lat = floatval($dist['dusun_lat']);
        $previous_lng = floatval($dist['dusun_lng']);
    }
    
    // Add return to depo distance
    $lat_diff = deg2rad($depo_lat - $previous_lat);
    $lng_diff = deg2rad($depo_lng - $previous_lng);
    $a = sin($lat_diff/2) * sin($lat_diff/2) +
         cos(deg2rad($previous_lat)) * cos(deg2rad($depo_lat)) *
         sin($lng_diff/2) * sin($lng_diff/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    $return_distance = $earth_radius * $c;
    $total_distance += $return_distance;
    
    $_SESSION['success'] = "Optimasi berhasil! Total jarak: " . round($total_distance, 2) . " km. Rute telah dioptimasi berdasarkan prioritas kondisi warga, jarak, dan efisiensi BBM.";
    redirect('optimasi.php');
}

// Get depo list
$stmt = $db->query("SELECT * FROM depo_pengiriman WHERE is_active = 1 ORDER BY nama_depo");
$depo_list = $stmt->fetchAll();

// Get distributions ready for optimization with prediction data
// Optimized query - removed subquery in SELECT to improve performance
$stmt = $db->query("SELECT dt.*, dus.nama_dusun, dus.latitude as dusun_lat, dus.longitude as dusun_lng,
                    des.nama_desa, d.nama_depo, d.latitude as depo_lat, d.longitude as depo_lng,
                    p.tingkat_risiko, p.skor_risiko
                    FROM distribusi_tangki dt
                    JOIN dusun dus ON dt.dusun_id = dus.id
                    JOIN desa des ON dus.desa_id = des.id
                    LEFT JOIN depo_pengiriman d ON dt.depo_id = d.id
                    LEFT JOIN prediksi_kekeringan p ON dus.id = p.dusun_id AND p.tanggal_prediksi >= CURDATE()
                    WHERE dt.status = 'disetujui'
                    ORDER BY dt.tanggal_jadwal, dt.urutan_rute
                    LIMIT 1000");
$distributions = $stmt->fetchAll();

// Get report counts separately for better performance (batch query)
if (!empty($distributions)) {
    $dusun_ids = array_unique(array_column($distributions, 'dusun_id'));
    $placeholders = str_repeat('?,', count($dusun_ids) - 1) . '?';
    $stmt = $db->prepare("SELECT dusun_id, COUNT(*) as jumlah_laporan 
                          FROM laporan_warga 
                          WHERE dusun_id IN ($placeholders) 
                          AND status = 'verified' 
                          AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                          GROUP BY dusun_id");
    $stmt->execute($dusun_ids);
    $report_counts = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Add report counts to distributions
    foreach ($distributions as &$dist) {
        $dist['jumlah_laporan'] = $report_counts[$dist['dusun_id']] ?? 0;
    }
    unset($dist);
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Optimasi Pengiriman - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
    /* PERMANENT CSS: Mencegah info AI hilang - PRIORITAS TERTINGGI */
    #ai-info-permanent,
    .ai-info-box-permanent {
        display: block !important;
        visibility: visible !important;
        opacity: 1 !important;
        position: relative !important;
        z-index: 999 !important;
        background: #d1ecf1 !important;
        border: 1px solid #bee5eb !important;
        border-radius: 5px !important;
        padding: 20px !important;
        margin-top: 20px !important;
        min-height: 200px !important;
    }
    
    #ai-info-permanent *,
    .ai-info-box-permanent * {
        visibility: visible !important;
    }
    
    #ai-info-permanent > div,
    .ai-info-box-permanent > div {
        display: grid !important;
        visibility: visible !important;
    }
    
    #ai-info-permanent > div > div,
    .ai-info-box-permanent > div > div {
        display: block !important;
        visibility: visible !important;
        opacity: 1 !important;
    }
    </style>
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fas fa-route"></i> AI Optimasi Pengiriman</h1>
                <p>Optimasi rute pengiriman berdasarkan jarak dan hemat BBM</p>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <!-- Optimize Form -->
        <div class="content-section">
            <h2><i class="fas fa-brain"></i> Optimasi Rute</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="optimize">
                <div class="form-row">
                    <div class="form-group">
                        <label>Tanggal Pengiriman</label>
                        <input type="date" name="tanggal" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Depo Pengiriman</label>
                        <select name="depo_id" required>
                            <option value="">Pilih Depo</option>
                            <?php foreach ($depo_list as $depo): ?>
                            <option value="<?php echo $depo['id']; ?>"><?php echo htmlspecialchars($depo['nama_depo']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-magic"></i> Optimasi Rute
                </button>
            </form>
            <!-- PERMANENT INFO: Cara Kerja AI - TIDAK AKAN HILANG - TIDAK MENGGUNAKAN CLASS ALERT -->
            <div id="ai-info-permanent" class="ai-info-box-permanent" style="margin-top: 20px; padding: 20px; background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 5px; display: block !important; visibility: visible !important; opacity: 1 !important; position: relative !important; z-index: 999 !important; min-height: 200px !important;">
                <h3 style="margin-top: 0; margin-bottom: 15px; color: #0c5460; display: block !important; visibility: visible !important;">
                    <i class="fas fa-brain"></i> Cara Kerja AI Optimasi Cerdas
                </h3>
                <div style="display: grid !important; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; visibility: visible !important;">
                    <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #e74c3c; display: block !important; visibility: visible !important; opacity: 1 !important;">
                        <strong style="color: #e74c3c;">1. Prioritas Kondisi Warga (50%)</strong><br>
                        <small>Daerah dengan status "Kering" mendapat prioritas tertinggi</small>
                    </div>
                    <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #3498db; display: block !important; visibility: visible !important; opacity: 1 !important;">
                        <strong style="color: #3498db;">2. Jarak dari Depo (30%)</strong><br>
                        <small>Daerah terdekat dari depo diprioritaskan untuk efisiensi</small>
                    </div>
                    <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #2ecc71; display: block !important; visibility: visible !important; opacity: 1 !important;">
                        <strong style="color: #2ecc71;">3. Laporan Warga (20%)</strong><br>
                        <small>Daerah dengan laporan lebih banyak mendapat prioritas</small>
                    </div>
                    <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #f39c12; display: block !important; visibility: visible !important; opacity: 1 !important;">
                        <strong style="color: #f39c12;">4. Optimasi Rute</strong><br>
                        <small>Setelah prioritas, rute dioptimasi untuk meminimalkan jarak tempuh</small>
                    </div>
                    <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #9b59b6; display: block !important; visibility: visible !important; opacity: 1 !important;">
                        <strong style="color: #9b59b6;">5. Hemat BBM</strong><br>
                        <small>Menghitung estimasi BBM (0.2 L/km) dan biaya pengiriman (Rp 10.000/L)</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Optimized Routes - PERMANENT, TIDAK AKAN HILANG -->
        <div class="content-section" id="optimized-routes-permanent" style="display: block !important; visibility: visible !important;">
            <h2><i class="fas fa-route"></i> Rute yang Telah Dioptimasi</h2>
            <p style="color: #666; margin-bottom: 20px; display: block !important;">
                <i class="fas fa-info-circle"></i> <strong>Tabel di bawah menampilkan semua rute yang telah dioptimasi dengan estimasi BBM (0.2 L/km) dan biaya pengiriman (Rp 10.000/L).</strong>
            </p>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Urutan</th>
                            <th>Tanggal</th>
                            <th>Desa/Dusun</th>
                            <th>Kondisi</th>
                            <th>Prioritas</th>
                            <th>Depo</th>
                            <th>Jarak (km)</th>
                            <th>Estimasi BBM (L)</th>
                            <th>Biaya BBM</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($distributions)): ?>
                        <tr>
                            <td colspan="10" class="text-center">Belum ada rute yang dioptimasi</td>
                        </tr>
                        <?php else: ?>
                        <?php 
                        $total_jarak = 0;
                        $total_bbm = 0;
                        $total_biaya = 0;
                        $earth_radius = 6371; // km
                        $first_depo_lat = null;
                        $first_depo_lng = null;
                        $last_dusun_lat = null;
                        $last_dusun_lng = null;
                        
                        foreach ($distributions as $index => $dist): 
                            // Use stored jarak_km from database (calculated during optimization)
                            $jarak_segment = floatval($dist['jarak_km'] ?? 0);
                            
                            // Store first depo coordinates for return calculation
                            if ($index === 0 && $dist['depo_lat'] && $dist['depo_lng']) {
                                $first_depo_lat = floatval($dist['depo_lat']);
                                $first_depo_lng = floatval($dist['depo_lng']);
                            }
                            
                            // Store last dusun coordinates for return calculation
                            if ($dist['dusun_lat'] && $dist['dusun_lng']) {
                                $last_dusun_lat = floatval($dist['dusun_lat']);
                                $last_dusun_lng = floatval($dist['dusun_lng']);
                            }
                            
                            $total_jarak += $jarak_segment;
                            $total_bbm += floatval($dist['estimasi_bbm_liter'] ?? 0);
                            $total_biaya += floatval($dist['biaya_bbm'] ?? 0);
                        ?>
                        <tr>
                            <td>
                                <?php if ($dist['urutan_rute'] > 0): ?>
                                <span class="badge badge-primary"><?php echo $dist['urutan_rute']; ?></span>
                                <?php else: ?>
                                <span class="badge badge-warning">-</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo formatDate($dist['tanggal_jadwal']); ?></td>
                            <td><?php echo htmlspecialchars($dist['nama_desa']); ?> - <?php echo htmlspecialchars($dist['nama_dusun']); ?></td>
                            <td>
                                <?php if ($dist['tingkat_risiko']): ?>
                                <span class="badge badge-<?php echo $dist['tingkat_risiko'] === 'kering' ? 'danger' : ($dist['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                                    <?php echo ucfirst($dist['tingkat_risiko']); ?>
                                </span>
                                <?php else: ?>
                                <span class="badge badge-info">Belum ada prediksi</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-primary" title="Skor Prioritas: <?php echo number_format($dist['priority_score'] ?? 0, 2); ?>">
                                    <?php echo number_format($dist['priority_score'] ?? 0, 1); ?>
                                </span>
                                <?php if (isset($dist['jumlah_laporan']) && $dist['jumlah_laporan'] > 0): ?>
                                <br><small style="color: #666;"><?php echo $dist['jumlah_laporan']; ?> laporan</small>
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($dist['nama_depo'] ?? '-'); ?></td>
                            <td>
                                <strong><?php echo number_format($jarak_segment, 2); ?> km</strong>
                                <?php if ($index === 0): ?>
                                <br><small style="color: #666;">dari depo</small>
                                <?php else: ?>
                                <br><small style="color: #666;">dari lokasi sebelumnya</small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php 
                                $bbm_value = floatval($dist['estimasi_bbm_liter'] ?? 0);
                                if ($bbm_value > 0): 
                                ?>
                                <strong style="color: #3498db;"><?php echo number_format($bbm_value, 2); ?> L</strong>
                                <br><small style="color: #666;">@ 0.2 L/km</small>
                                <?php else: ?>
                                <span style="color: #999;">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php 
                                $biaya_value = floatval($dist['biaya_bbm'] ?? 0);
                                if ($biaya_value > 0): 
                                ?>
                                <strong style="color: #e74c3c;">Rp <?php echo number_format($biaya_value, 0, ',', '.'); ?></strong>
                                <br><small style="color: #666;">@ Rp 10.000/L</small>
                                <?php else: ?>
                                <span style="color: #999;">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-success">Disetujui</span>
                            </td>
                        </tr>
                        <?php 
                        endforeach; 
                        
                        // Calculate return distance to depo
                        $return_distance = 0;
                        if ($last_dusun_lat && $last_dusun_lng && $first_depo_lat && $first_depo_lng) {
                            $lat_diff = deg2rad($first_depo_lat - $last_dusun_lat);
                            $lng_diff = deg2rad($first_depo_lng - $last_dusun_lng);
                            $a = sin($lat_diff/2) * sin($lat_diff/2) +
                                 cos(deg2rad($last_dusun_lat)) * cos(deg2rad($first_depo_lat)) *
                                 sin($lng_diff/2) * sin($lng_diff/2);
                            $c = 2 * atan2(sqrt($a), sqrt(1-$a));
                            $return_distance = $earth_radius * $c;
                        }
                        
                        $total_jarak_dengan_kembali = $total_jarak + $return_distance;
                        $bbm_kembali = $return_distance * 0.2;
                        $biaya_kembali = $bbm_kembali * 10000;
                        $total_bbm_dengan_kembali = $total_bbm + $bbm_kembali;
                        $total_biaya_dengan_kembali = $total_biaya + $biaya_kembali;
                        ?>
                        <?php if (count($distributions) > 0): ?>
                        <tr style="background: #e8f4f8; font-weight: bold; border-top: 2px solid #3498db;">
                            <td colspan="6" style="text-align: right; padding-right: 20px;">
                                <strong><i class="fas fa-calculator"></i> Subtotal (Perjalanan ke Lokasi):</strong>
                            </td>
                            <td><strong><?php echo number_format($total_jarak, 2); ?> km</strong></td>
                            <td><strong><?php echo number_format($total_bbm, 2); ?> L</strong></td>
                            <td><strong>Rp <?php echo number_format($total_biaya, 0, ',', '.'); ?></strong></td>
                            <td></td>
                        </tr>
                        <?php 
                        // Calculate return distance to depo
                        $return_distance = 0;
                        $bbm_kembali = 0;
                        $biaya_kembali = 0;
                        if ($last_dusun_lat && $last_dusun_lng && $first_depo_lat && $first_depo_lng) {
                            $lat_diff = deg2rad($first_depo_lat - $last_dusun_lat);
                            $lng_diff = deg2rad($first_depo_lng - $last_dusun_lng);
                            $a = sin($lat_diff/2) * sin($lat_diff/2) +
                                 cos(deg2rad($last_dusun_lat)) * cos(deg2rad($first_depo_lat)) *
                                 sin($lng_diff/2) * sin($lng_diff/2);
                            $c = 2 * atan2(sqrt($a), sqrt(1-$a));
                            $return_distance = $earth_radius * $c;
                            $bbm_kembali = $return_distance * 0.2;
                            $biaya_kembali = $bbm_kembali * 10000;
                        }
                        
                        $total_jarak_dengan_kembali = $total_jarak + $return_distance;
                        $total_bbm_dengan_kembali = $total_bbm + $bbm_kembali;
                        $total_biaya_dengan_kembali = $total_biaya + $biaya_kembali;
                        
                        if ($return_distance > 0): 
                        ?>
                        <tr style="background: #fff3cd; font-weight: bold;">
                            <td colspan="6" style="text-align: right; padding-right: 20px;">
                                <strong><i class="fas fa-undo"></i> Kembali ke Depo:</strong>
                            </td>
                            <td><strong><?php echo number_format($return_distance, 2); ?> km</strong></td>
                            <td><strong><?php echo number_format($bbm_kembali, 2); ?> L</strong></td>
                            <td><strong>Rp <?php echo number_format($biaya_kembali, 0, ',', '.'); ?></strong></td>
                            <td></td>
                        </tr>
                        <?php endif; ?>
                        <tr style="background: #d4edda; font-weight: bold; border-top: 3px solid #28a745; font-size: 1.1em;">
                            <td colspan="6" style="text-align: right; padding-right: 20px;">
                                <strong style="color: #155724;"><i class="fas fa-check-circle"></i> TOTAL KESELURUHAN:</strong>
                            </td>
                            <td style="color: #155724;">
                                <strong><?php echo number_format($total_jarak_dengan_kembali, 2); ?> km</strong>
                            </td>
                            <td style="color: #155724;">
                                <strong><?php echo number_format($total_bbm_dengan_kembali, 2); ?> L</strong>
                            </td>
                            <td style="color: #155724;">
                                <strong>Rp <?php echo number_format($total_biaya_dengan_kembali, 0, ',', '.'); ?></strong>
                            </td>
                            <td></td>
                        </tr>
                        <?php endif; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <script src="../assets/js/main.js"></script>
    <script>
    // PERMANENT: Pastikan informasi AI dan tabel tidak hilang
    (function() {
        'use strict';
        
        function ensureVisible() {
            // Pastikan info AI selalu terlihat - FORCE dengan multiple methods
            const aiInfo = document.getElementById('ai-info-permanent');
            if (aiInfo) {
                // Remove any hide classes
                aiInfo.classList.remove('hide', 'fade', 'fade-out', 'd-none', 'hidden', 'alert-info');
                aiInfo.classList.add('ai-info-box-permanent');
                
                // Force dengan cssText untuk override semua style
                aiInfo.style.cssText = 'display: block !important; visibility: visible !important; opacity: 1 !important; position: relative !important; z-index: 999 !important; background: #d1ecf1 !important; border: 1px solid #bee5eb !important; border-radius: 5px !important; padding: 20px !important; margin-top: 20px !important; min-height: 200px !important;';
                
                // Force semua children visible (optimized - hanya check sekali, tidak setiap kali)
                if (!aiInfo.dataset.childrenProcessed) {
                    const allChildren = aiInfo.querySelectorAll('*');
                    allChildren.forEach(function(el) {
                        el.style.setProperty('display', '', 'important');
                        el.style.setProperty('visibility', 'visible', 'important');
                        el.style.setProperty('opacity', '1', 'important');
                        el.classList.remove('hide', 'fade', 'fade-out', 'd-none', 'hidden');
                    });
                    aiInfo.dataset.childrenProcessed = 'true';
                }
            }
            
            // Pastikan tabel rute selalu terlihat
            const routesSection = document.getElementById('optimized-routes-permanent');
            if (routesSection) {
                routesSection.style.setProperty('display', 'block', 'important');
                routesSection.style.setProperty('visibility', 'visible', 'important');
            }
        }
        
        // Jalankan saat DOM ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', ensureVisible);
        } else {
            ensureVisible();
        }
        
        // Jalankan saat window load
        window.addEventListener('load', ensureVisible);
        
        // Monitor perubahan dan pastikan tetap terlihat (optimized)
        const aiInfo = document.getElementById('ai-info-permanent');
        if (aiInfo) {
            let lastCheck = 0;
            const observer = new MutationObserver(function() {
                // Throttle: hanya check setiap 500ms untuk prevent too many calls
                const now = Date.now();
                if (now - lastCheck > 500) {
                    lastCheck = now;
                    ensureVisible();
                }
            });
            observer.observe(aiInfo, { 
                attributes: true, 
                attributeFilter: ['style', 'class'],
                childList: false, // Disable childList untuk reduce overhead
                subtree: false // Disable subtree untuk reduce overhead
            });
        }
        
        // Check setiap 2 detik untuk memastikan tetap terlihat (tidak terlalu sering)
        // Reduced frequency to prevent performance issues
        let checkInterval = setInterval(ensureVisible, 2000);
        
        // Stop checking after 30 seconds to prevent infinite checking
        setTimeout(function() {
            clearInterval(checkInterval);
        }, 30000);
        
        // Override jQuery hide/fade jika ada
        if (typeof jQuery !== 'undefined') {
            const originalHide = jQuery.fn.hide;
            const originalFadeOut = jQuery.fn.fadeOut;
            
            jQuery.fn.hide = function() {
                if (this.attr('id') === 'ai-info-permanent' || this.closest('#ai-info-permanent').length > 0) {
                    return this; // Don't hide
                }
                return originalHide.apply(this, arguments);
            };
            
            jQuery.fn.fadeOut = function() {
                if (this.attr('id') === 'ai-info-permanent' || this.closest('#ai-info-permanent').length > 0) {
                    return this; // Don't fade out
                }
                return originalFadeOut.apply(this, arguments);
            };
        }
        
        // Prevent removeChild on our element
        const originalRemoveChild = Node.prototype.removeChild;
        Node.prototype.removeChild = function(child) {
            if (child && (child.id === 'ai-info-permanent' || (child.querySelector && child.querySelector('#ai-info-permanent')))) {
                console.log('Blocked removal of AI info');
                return child;
            }
            return originalRemoveChild.apply(this, arguments);
        };
    })();
    </script>
</body>
</html>

